#!/bin/bash

# --- Configuration ---
SERVICE_NAME="B.service"
ERROR_STRING="tunnel server dialer: dial tcp" # Partial string to look for in logs
LOG_FILE="/var/log/E1.log" # Log file for this script's actions
MAX_RESTARTS_IN_ROW=1 # Maximum consecutive restarts to prevent infinite loops
CHECK_INTERVAL_SECONDS=10 # How often to check logs for errors (in seconds)
SCHEDULED_RESTART_INTERVAL_MINUTES=30 # How often to perform a scheduled restart (in minutes)
RESTART_STABILIZE_SLEEP=10 # Time to wait after a restart for service to stabilize

# --- Functions ---

# Function to log messages with a timestamp
log_message() {
    echo "$(date '+%Y-%m-%d %H:%M:%S') - $1" | tee -a "$LOG_FILE"
}

# Function to restart the service
restart_backhaul() {
    log_message "Initiating restart of $SERVICE_NAME."

    # Restart the backhaul service
    log_message "Restarting $SERVICE_NAME."
    sudo systemctl restart "$SERVICE_NAME"
    if [ $? -eq 0 ]; then
        log_message "$SERVICE_NAME restarted successfully."
        sleep "$RESTART_STABILIZE_SLEEP" # Give service time to start and log
        return 0 # Success
    else
        log_message "Failed to restart $SERVICE_NAME. Check systemctl status. Error code: $?."
        return 1 # Failure
    fi
}

# --- Main Script Logic ---

# Ensure log file exists and is writable
if ! touch "$LOG_FILE" 2>/dev/null; then
    echo "Error: Cannot create or write to log file: $LOG_FILE. Exiting."
    exit 1
fi

log_message "Starting monitoring and restart script for $SERVICE_NAME..."

restart_count=0
last_scheduled_restart_time=$(date +%s) # Initialize with current time in seconds since epoch

while true; do
    current_time=$(date +%s)
    
    # Check for scheduled restart
    if (( current_time - last_scheduled_restart_time >= SCHEDULED_RESTART_INTERVAL_MINUTES * 60 )); then
        log_message "Performing scheduled restart (every $SCHEDULED_RESTART_INTERVAL_MINUTES minutes)."
        restart_backhaul
        if [ $? -ne 0 ]; then
            log_message "Scheduled restart failed. Script will continue to monitor."
        fi
        last_scheduled_restart_time=$current_time # Update last scheduled restart time
        restart_count=0 # Reset error-based restart count after a scheduled restart
    fi

    log_message "Checking for error string: '$ERROR_STRING' in $SERVICE_NAME logs..."

    # Use journalctl to get the last few lines of the service log
    if journalctl -u "$SERVICE_NAME" --no-pager -n 5 | grep -q "$ERROR_STRING"; then
        log_message "ERROR '$ERROR_STRING' DETECTED in $SERVICE_NAME logs!"

        restart_count=$((restart_count + 1))
        log_message "Error-based restart attempt #$restart_count."

        if [ "$restart_count" -gt "$MAX_RESTARTS_IN_ROW" ]; then
            log_message "Maximum consecutive error-based restarts ($MAX_RESTARTS_IN_ROW) reached. Aborting."
            log_message "Manual intervention required for $SERVICE_NAME."
            exit 1
        fi

        # Call the restart function
        restart_backhaul
        if [ $? -ne 0 ]; then
            log_message "Error-based restart failed. Exiting script."
            exit 1
        fi

    else
        # If no error detected, reset the error-based restart counter
        if [ "$restart_count" -gt 0 ]; then
            log_message "No error detected in the last check. Resetting consecutive error-based restart count."
        fi
        restart_count=0
        log_message "No error detected. $SERVICE_NAME appears to be running normally."
    fi

    # Wait before the next check for errors
    sleep "$CHECK_INTERVAL_SECONDS"
done
